<?php
/**
 * Setup Member Types Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Member Types Component Class
 */
class Member_Types extends Component {
	/**
	 * Constructor
	 */
	public function __construct() {
		$path = trailingslashit( dirname( __FILE__ ) );

		parent::start(
			'member-types',
			esc_html__( 'Member Types', 'nisje' ),
			$path,
			'member_types',
			'dekode_intranet_settings_member_types'
		);
	}

	/**
	 * Global variables
	 */
	public function setup_globals() {
		$settings = get_option( $this->option_key );
		if ( is_array( $settings ) && ! empty( $settings ) ) {
			foreach ( $settings as $key => $value ) {
				$this->settings[ $key ] = $value;
			}
		}
	}

	/**
	 * Setup Filters
	 */
	public function setup_filters() {
		// Register BuddyPress member types.
		add_action( 'bp_register_member_types', [ $this, 'register_member_types' ] );

		if ( is_admin() ) {
			add_action( 'nisje_settings_register_field_groups', [ $this, 'register_field_group' ] );
			add_action( 'nisje_settings_add_option_page', [ $this, 'setup_settings_menu' ] );
			add_action( 'nisje_settings_populate_options_from_acf', [ $this, 'populate_acf_settings' ] );
		}

		add_filter( 'nisje_custom_profile_layouts', [ $this, 'register_profile_layout' ] );
		add_filter( 'nisje_custom_profile_layout_value_member_type', [ $this, 'register_profile_layout_value' ], 10, 2 );
	}

	/**
	 * Register REST Endpoints
	 */
	public function register_rest_routes() {
		nisje_include( 'includes/components/member-types/rest-handlers/extend-settings.php' );
		nisje_include( 'includes/components/member-types/rest-handlers/extend-user.php' );
	}

	/**
	 * Register member types
	 */
	public function register_member_types() {
		$member_types = nisje_get_setting( 'member_types' );

		if ( ! empty( $member_types ) ) {
			foreach ( $member_types as $member_key => $args ) {
				if ( ! empty( $member_key ) && is_array( $args ) && ! empty( $args ) ) {
					bp_register_member_type( $member_key, $args );

					// translators: %s Menu name.
					register_nav_menu( 'primary-' . $member_key, sprintf( __( 'Primary Menu (%s)', 'nisje' ), $args['labels']['name'] ) );

					// translators: %s Menu name.
					register_nav_menu( 'shortcuts-' . $member_key, sprintf( __( 'Shortcuts Menu (%s)', 'nisje' ), $args['labels']['name'] ) );
				}
			}
		}
	}

	/**
	 * Setup options field group.
	 */
	public function register_field_group() {
		nisje_include( 'includes/components/member-types/acf/field-groups/acf-field-group-options-member-types.php' );
	}

	/**
	 * Setup settings menu.
	 */
	public function setup_settings_menu() {
		acf_add_options_sub_page( [
			'page_title'  => esc_html__( 'Member Types', 'nisje' ),
			'menu_title'  => esc_html__( 'Member Types', 'nisje' ),
			'capability'  => nisje_get_setting( 'settings_cap', 'core' ),
			'menu_slug'   => 'nisje-settings-member-types',
			'parent_slug' => nisje_get_setting( 'settings_key', 'core' ),
		] );
	}

	/**
	 * Populate ACF Options Member Types
	 *
	 * @param WP_Screen $screen Screen object.
	 */
	public function populate_acf_settings( $screen ) {
		if ( false !== strpos( $screen->id, 'nisje-settings-member-types' ) ) {
			$new_member_types = [];

			$member_types = get_field( 'dekode_intranet_member_types', 'option' );
			if ( ! empty( $member_types ) ) {
				foreach ( $member_types as $member_type ) {
					$key            = '';
					$name           = '';
					$singular_name  = '';
					$directory_slug = false;

					if ( isset( $member_type['dekode_intranet_member_types_key'] ) && ! empty( $member_type['dekode_intranet_member_types_key'] ) ) {
						$key = sanitize_key( $member_type['dekode_intranet_member_types_key'] );
					}

					if ( isset( $member_type['dekode_intranet_member_types_name'] ) && ! empty( $member_type['dekode_intranet_member_types_name'] ) ) {
						$name = esc_html( $member_type['dekode_intranet_member_types_name'] );
					}

					if ( isset( $member_type['dekode_intranet_member_types_singular_name'] ) && ! empty( $member_type['dekode_intranet_member_types_singular_name'] ) ) {
						$singular_name = esc_html( $member_type['dekode_intranet_member_types_singular_name'] );
					}

					if ( isset( $member_type['dekode_intranet_member_types_has_directory'] ) && $member_type['dekode_intranet_member_types_has_directory'] ) {
						if ( isset( $member_type['dekode_intranet_member_types_directory_slug'] ) && ! empty( $member_type['dekode_intranet_member_types_directory_slug'] ) ) {
							$directory_slug = sanitize_key( $member_type['dekode_intranet_member_types_directory_slug'] );
						}
					}

					if ( $key && $name && $singular_name ) {
						$new_member_types[ $key ] = [
							'labels'        => [
								'name'          => $name,
								'singular_name' => $singular_name,
							],
							'has_directory' => $directory_slug,
						];
					}
				}
			}

			update_option( $this->option_key, $new_member_types );
		}
	}

	/**
	 * Register profile layout.
	 *
	 * @param array $layouts Custom layouts.
	 * @return array $layouts Optimized layouts.
	 */
	public function register_profile_layout( $layouts ) {
		$layouts['member_type'] = esc_html__( 'Member Type', 'nisje' );

		return $layouts;
	}

	/**
	 * Register profile layout value.
	 *
	 * @param array $field   Field data.
	 * @param int   $user_id The user id.
	 *
	 * @return array Value.
	 */
	public function register_profile_layout_value( $field, $user_id ) {
		$retval = bp_get_member_type( $user_id, true );
		if ( $retval ) {
			$type_object = bp_get_member_type_object( $retval );
			if ( $type_object && isset( $type_object->labels['name'] ) ) {
				$retval = $type_object->labels['name'];
			}

			return [
				'name'  => esc_html__( 'Member Type', 'nisje' ),
				'value' => $retval,
				'type'  => 'member_type',
			];
		}
	}
}
