<?php
/**
 * Extend Post Types
 *
 * @package Nisje
 */

defined( 'ABSPATH' ) || die( 'Shame on you' );

// Types to extend.
$post_types = apply_filters( 'nisje_rest_shared_fields_post_types', [] );

if ( empty( $post_types ) || ! is_array( $post_types ) ) {
	return;
}

// Comment count.
register_rest_field(
	$post_types, 'comment_count', [
		'get_callback' => function ( $object ) {
			$comments = wp_count_comments( $object['id'] );

			return $comments->total_comments;
		},
		'schema'       => [
			'description' => esc_html__( 'Number of comments', 'nisje' ),
			'type'        => 'integer',
			'readonly'    => true,
		],
	]
);

// Comments.
register_rest_field(
	$post_types, 'comments', [
		'get_callback' => function ( $object ) {
			$request = new WP_REST_Request( 'GET', '/wp/v2/comments' );

			$request->set_query_params( [ 'post' => $object['id'] ] );

			$response = rest_do_request( $request );
			$server   = rest_get_server();
			$data     = $server->response_to_data( $response, false );

			return array_reverse( $data, false );
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Comments object', 'nisje' ),
			'type'        => 'object',
			'readonly'    => true,
		],
	]
);

register_rest_field(
	$post_types, 'author_name', [
		'get_callback' => function( $object ) {
			$retval = '';

			$post = get_post( $object['id'] );
			if ( $post instanceof \WP_Post ) {
				$retval = bp_core_get_user_displayname( $post->post_author );
			}

			return $retval;
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Author Name', 'nisje' ),
			'type'        => 'string',
			'readonly'    => true,
		],
	]
);

// Image data.
register_rest_field(
	$post_types, 'image', [
		'get_callback' => function( $object ) {
			if ( has_post_thumbnail( $object['id'] ) ) {
				return nisje_populate_image_data( get_post_thumbnail_id( $object['id'] ) );
			}

			return [];
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Image data', 'nisje' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'string',
			],
			'readonly'    => true,
		],
	]
);

// Reactions.
register_rest_field(
	$post_types, 'reactions', [
		'get_callback' => function ( $object ) {
			return nisje_get_user_reactions( $object['id'], 'post' );
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Reactions', 'nisje' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'mixed',
			],
			'readonly'    => true,
		],
	]
);

// Attachments.
register_rest_field(
	$post_types, 'attachments', [
		'get_callback'    => 'nisje_get_post_attachments',
		'update_callback' => 'nisje_update_post_attachments',
		'schema'          => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'IDs of attachments', 'nisje' ),
			'type'        => 'array',
			'items'       => [
				'type' => 'integer',
			],
			'readonly'    => false,
		],
	]
);

// Add avatar URLs.
register_rest_field(
	$post_types, 'avatar_urls', [
		'get_callback' => function( $object ) {
			$post = get_post( $object['id'] );
			if ( $post instanceof \WP_Post ) {
				return nisje_get_user_avatar( $post->post_author );
			}

			return [];
		},
		'schema'       => [
			'context'     => [ 'view', 'edit' ],
			'description' => esc_html__( 'Avatar URLs for the resource', 'nisje' ),
			'type'        => 'object',
			'properties'  => nisje_get_avatar_rest_properties(),
			'readonly'    => true,
		],
	]
);
