<?php
/**
 * Settings endpoints.
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components\Rest;

defined( 'ABSPATH' ) || die( 'Shame on you' );

/**
 * Settings Rest Class.
 */
class Settings_Controller extends \WP_REST_Controller {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->namespace = 'nisje/v1';
		$this->rest_base = 'settings';
	}
	/**
	 * Register the routes.
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/',
			[
				[
					'methods'  => \WP_REST_Server::READABLE,
					'callback' => [ $this, 'get_item' ],
					'args'     => [
						'context' => $this->get_context_param(
							[
								'default' => 'view',
							]
						),
					],
				],
				'schema' => [ $this, 'get_public_item_schema' ],
			]
		);
	}
	/**
	 * Retrieve graph data from a website.
	 *
	 * @param  WP_REST_Request $request Request.
	 * @return WP_REST_Request|WP_Error Plugin object data on success, WP_Error otherwise.
	 */
	public function get_item( $request ) {
		$settings = [];

		$settings['name'] = get_bloginfo( 'name' );

		$logo = get_theme_mod( 'custom_logo' );
		if ( $logo ) {
			$settings['logo'] = wp_get_attachment_url( $logo );
		} else {
			$settings['logo'] = false;
		}

		$settings['login_info_message'] = nisje_get_setting( 'login_info_message', 'general' );

		$settings['skip_second_registration'] = nisje_get_setting( 'skip_second_registration', 'general' );
		$settings['hide_forgotten_password']  = nisje_get_setting( 'hide_forgotten_password', 'general' );

		$settings['terms'] = nisje_get_setting( 'terms', 'general' );

		$settings['lostpassword_url'] = wp_lostpassword_url( home_url( '/' ) );

		$data = $this->prepare_item_for_response( $settings, $request );

		$response = rest_ensure_response( $data );

		return $response;
	}

	/**
	 * Prepares scraper data for return as an object.
	 *
	 * @param stdClass        $item    Item data.
	 * @param WP_REST_Request $request Request.
	 * @param boolean         $is_raw  Optional, not used. Defaults to false.
	 *
	 * @return WP_REST_Response
	 */
	public function prepare_item_for_response( $item, $request, $is_raw = false ) {
		$schema = $this->get_item_schema();

		$data = [];

		if ( ! empty( $schema['properties']['name'] ) ) {
			$data['name'] = esc_html( $item['name'] );
		}

		if ( ! empty( $schema['properties']['logo'] ) ) {
			$data['logo'] = esc_html( $item['logo'] );
		}

		if ( ! empty( $schema['properties']['skip_second_registration'] ) ) {
			$data['skip_second_registration'] = $item['skip_second_registration'];
		}

		if ( ! empty( $schema['properties']['hide_forgotten_password'] ) ) {
			$data['hide_forgotten_password'] = $item['hide_forgotten_password'];
		}

		if ( ! empty( $schema['properties']['login_info_message'] ) ) {
			$data['login_info_message'] = $item['login_info_message'];
		}

		if ( ! empty( $schema['properties']['terms'] ) ) {
			$data['terms'] = $item['terms'];
		}

		if ( ! empty( $schema['properties']['lostpassword_url'] ) ) {
			$data['lostpassword_url'] = esc_url( $item['lostpassword_url'] );
		}

		if ( ! empty( $schema['properties']['users_can_register'] ) ) {
			$data['users_can_register'] = (bool) get_option( 'users_can_register' );
		}

		if ( ! empty( $schema['properties']['active_components'] ) ) {
			$data['active_components'] = nisje_get_active_components();
		}

		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->add_additional_fields_to_object( $data, $request );
		$data    = $this->filter_response_by_context( $data, $context );

		$response = rest_ensure_response( $data );

		/**
		 * Filter an scraper value returned from the API.
		 *
		 * @param array           $response
		 * @param WP_REST_Request $request Request used to generate the response.
		 */
		return apply_filters( 'nisje_rest_prepare_settings_value', $response, $item, $request );
	}
	/**
	 * Get the plugin schema, conforming to JSON Schema.
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$schema = [
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => 'settings',
			'type'       => 'object',
			'properties' => [
				'name'                     => [
					'description' => esc_html__( 'WP site title', 'nisje' ),
					'type'        => 'string',
					'readonly'    => true,
				],
				'logo'                     => [
					'description' => esc_html__( 'Logo', 'nisje' ),
					'type'        => 'string',
					'readonly'    => true,
				],
				'skip_second_registration' => [
					'description' => esc_html__( 'Skip second registration screen', 'nisje' ),
					'type'        => 'boolean',
					'readonly'    => true,
				],
				'hide_forgotten_password'  => [
					'description' => esc_html__( 'Forgotten password link is hidden', 'nisje' ),
					'type'        => 'boolean',
					'readonly'    => true,
				],
				'login_info_message'       => [
					'description' => esc_html__( 'Login info message text', 'nisje' ),
					'type'        => 'string',
					'readonly'    => true,
				],
				'terms'                    => [
					'description' => esc_html__( 'The terms page post object', 'nisje' ),
					'type'        => 'object',
					'readonly'    => true,
				],
				'lostpassword_url'         => [
					'description' => esc_html__( 'Url to template that allows users to retrieve lost password', 'nisje' ),
					'type'        => 'string',
					'readonly'    => true,
				],
				'users_can_register'       => [
					'description' => esc_html__( 'Allow registration?', 'nisje' ),
					'type'        => 'boolean',
					'readonly'    => true,
				],
				'active_components'        => [
					'description' => esc_html__( 'Active components', 'nisje' ),
					'type'        => 'array',
					'items'       => [
						'type' => 'string',
					],
					'readonly'    => true,
				],
			],
		];

		return $this->add_additional_fields_schema( $schema );
	}
}
