<?php
/**
 * Setup User Content Component
 *
 * @package Nisje
 */

namespace Dekode\Nisje\Components;

defined( 'ABSPATH' ) || die( 'Shame on you' );

if ( ! class_exists( '\\Dekode\\Nisje\\Components\\User_Content' ) && class_exists( '\\Dekode\\Nisje\\Components\\Component' ) ) {

	/**
	 * User Content Component Class
	 */
	class User_Content extends Component {
		/**
		 * Constructor
		 */
		public function __construct() {
			$path = trailingslashit( dirname( __FILE__ ) );
			$name = 'user-content';

			parent::start(
				$name,
				esc_html__( 'User Content', 'nisje-user-content' ),
				$path,
				$name
			);
		}

		/**
		 * Global variables
		 */
		public function setup_globals() {
			$defaults = [
				// Public.
				'capability_key'                => 'nisje_user_content',
				'post_type_name'                => 'nisje-user-content',
				'post_type_rest_endpoint'       => 'nisje-user-content',
				'taxonomy_format_name'          => 'nisje-user-content-format',
				'taxonomy_format_rest_endpoint' => 'nisje-user-content-format',
				'taxonomy_type_name'            => 'nisje-user-content-type',
				'taxonomy_type_rest_endpoint'   => 'nisje-user-content-type',
				// Formats.
				'formats'                       => [
					'image'      => esc_html__( 'Image', 'nisje-user-content' ),
					'video'      => esc_html__( 'Video', 'nisje-user-content' ),
					'gallery'    => esc_html__( 'Gallery', 'nisje-user-content' ),
					'link'       => esc_html__( 'Link', 'nisje-user-content' ), // Share.
					'attachment' => esc_html__( 'Attachment', 'nisje-user-content' ),
				],
				// Types.
				'types'                         => [
					'social'     => esc_html__( 'Social Post', 'nisje-user-content' ),
					'discussion' => esc_html__( 'Discussion', 'nisje-user-content' ),
					'question'   => esc_html__( 'Question', 'nisje-user-content' ),
					'resources'  => esc_html__( 'Resource', 'nisje-user-content' ),
					'experience' => esc_html__( 'Experience', 'nisje-user-content' ),
				],
				// General.
				'images_meta_key'               => 'nisje_images',
			];

			$this->settings = wp_parse_args( $this->settings, $defaults );

			$this->public_settings = [
				'types'   => $this->settings['types'],
				'formats' => $this->settings['formats'],
			];
		}

		/**
		 * Setup filters
		 */
		public function setup_filters() {
			add_filter( 'nisje_feed_post_types', [ $this, 'add_feed_post_types' ] );
			add_action( 'nisje_update_post_attachments', [ $this, 'map_attachment_tax_format' ], 10, 2 );
			add_filter( 'nisje_taxonomy_tags_object_types', [ $this, 'add_global_tag_support' ] );

			add_filter( 'nisje_rest_shared_fields_group_post_types', [ $this, 'add_shared_rest_group_fields' ] );
			add_filter( 'nisje_rest_shared_fields_post_types', [ $this, 'add_shared_rest_fields' ] );
		}

		/**
		 * Register User Content post types and taxonomies.
		 */
		public function register_content_types() {
			$this->include_file( 'taxonomies/taxonomy-nisje-user-content-type.php' );
			$this->include_file( 'taxonomies/taxonomy-nisje-user-content-format.php' );
			$this->include_file( 'post-types/post-type-nisje-user-content.php' );
		}

		/**
		 * Register REST Endpoints
		 */
		public function register_rest_routes() {
			$this->include_file( 'rest-handlers/extend-user-content.php' );
			$this->include_file( 'rest-handlers/class-user-content-controller.php' );
		}

		/**
		 * Add feed post types.
		 *
		 * @param array $post_types Current Post Types.
		 * @return array $post_types Populated Post Types.
		 */
		public function add_feed_post_types( $post_types ) {
			$post_types[] = nisje_get_setting( 'post_type_name', 'user-content' );

			return $post_types;
		}

		/**
		 * Extend rest group types.
		 *
		 * @param array $post_types Current Post Types.
		 * @return array $post_types Populated Post Types.
		 */
		public function add_shared_rest_group_fields( $post_types ) {
			$post_types[] = nisje_get_setting( 'post_type_rest_endpoint', 'user-content' );

			return $post_types;
		}

		/**
		 * Extend rest post types.
		 *
		 * @param array $post_types Current Post Types.
		 * @return array $post_types Populated Post Types.
		 */
		public function add_shared_rest_fields( $post_types ) {
			$post_types[] = nisje_get_setting( 'post_type_rest_endpoint', 'user-content' );

			return $post_types;
		}

		/**
		 * Map taxonomy post formats when inserting attachments.
		 *
		 * @param mixed   $field  The object from the response.
		 * @param WP_Post $object The object from the response.
		 */
		public function map_attachment_tax_format( $field, $object ) {
			if ( $object instanceof \WP_Post ) {
				wp_set_post_terms( $object->ID, 'attachment', nisje_get_setting( 'taxonomy_format_name', 'user-content' ) );
			}
		}

		/**
		 * Add global tag support.
		 *
		 * @param array $post_types Current Post Types.
		 * @return array $post_types Populated Post Types.
		 */
		public function add_global_tag_support( $post_types ) {
			$post_types[] = nisje_get_setting( 'post_type_name', 'user-content' );

			return $post_types;
		}
	}
}
